/* 
 *      XAB 1.0
 *
 *		Adam Beguelin
 *		Carnegie Mellon University
 *		Pittsburgh Supercomputing Center
 *      26 Nov 1991
 *
 *      ablib.c
 *		
 *		Monitor event calls for xab
 *
 * These calls generate sends back to the xab process which 
 * displays them for the user.
 *
 */

/* Changes
 * 05/07/92 Added size to XAB_snd,  adamb
 */

#include "both.h"
#include "ab.h"
#include <stdio.h>
#include <sys/param.h>
#include <sys/time.h>

int xabevent;
int xab_msize;
static int snd_serial = 0;
static int rcv_serial = -1;
static mdesc cur_msg;


/***********************
* Crunch related calls *
***********************/

void xab_initsend()
{
	/* tell xab about this call */
	xabevent = XAB_initsend;
	initsend();
	snd_serial++;
	xab_msize = 0;

	xab_puttime();
	putnint(&xabevent, 1);
	putnint(&snd_serial, 1);
	snd(XAB, 0, XABMTYPE);

	/* do the init send but put a serial number at the head */
	initsend();
	putnint(&snd_serial, 1);


}

/*
*	xab_putnint
*	call putnint
*	add a message header to the current message header list
*/

int
xab_putnint(np, cnt)
    int *np;
	int cnt;
{
	int tmp;

	xabevent = XAB_putnint;
	tmp = putnint(np, cnt);
	xab_msize += cnt * sizeof(int);

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
	    xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_getnint(np, cnt)
    int *np;
	int cnt;
{

	int rc;

	xabevent = XAB_getnint;
	rc = getnint(np,cnt);

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_putnshort(np, cnt)
    short *np;
	int cnt;
{
	int tmp;

	xabevent = XAB_putnshort;
	tmp = putnshort(np, cnt);
	xab_msize += cnt * sizeof(short);

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_getnshort(np, cnt)
    short *np;
	int cnt;
{
	int rc;

	xabevent = XAB_getnshort;
	rc = getnshort(np, cnt);

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_putnlong(np, cnt)
    long *np;
	int cnt;
{
	int tmp;

	xabevent = XAB_putnlong;
	tmp = putnlong(np, cnt);
	xab_msize += cnt * sizeof(long);

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_getnlong(np, cnt)
    long *np;
	int cnt;
{
	int rc;

	xabevent = XAB_getnlong;
	rc = getnlong(np, cnt);

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_putnfloat(np, cnt)
    float *np;
	int cnt;
{
	int tmp;

	xabevent = XAB_putnfloat;
	tmp = putnfloat(np, cnt);
	xab_msize += cnt * sizeof(float);

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_getnfloat(fp, cnt)
    float *fp;
	int cnt;
{
	int rc;

	xabevent = XAB_getnfloat;
	rc = getnfloat(fp,cnt);

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}


int
xab_putndfloat(dp, cnt)
    double *dp;
	int cnt;
{
	int tmp;

	xabevent = XAB_putndfloat;
	tmp = putndfloat(dp, cnt);
	xab_msize += cnt * sizeof(double);

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_getndfloat(dp, cnt)
    double *dp;
	int cnt;
{
	int rc;

	xabevent = XAB_getndfloat;
	rc = (getndfloat(dp, cnt));

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_putncplx(xp, cnt)
    float *xp;
	int cnt;
{
	int tmp;

	xabevent = XAB_putncplx;
	tmp = putncplx(xp, cnt);
	xab_msize += cnt * sizeof(float);

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_getncplx(xp, cnt)
    float *xp;
	int cnt;
{
	int rc;

	xabevent = XAB_getncplx;
	rc = (getncplx(xp, cnt));

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_putbytes(cp, cnt)
    char *cp;
	int cnt;
{
	int tmp;

	xabevent = XAB_putbytes;
	tmp = putbytes(cp, cnt);
	xab_msize += cnt;

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_getbytes(cp, cnt)
    char *cp;
	int cnt;
{
	int rc;

	xabevent = XAB_getbytes;
	rc = getbytes(cp, cnt);

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_putstring(cp)
    char *cp;
{
	int tmp, cnt;

	cnt = strlen(cp)+1;
	xabevent = XAB_putstring;
	tmp = putstring(cp);
	xab_msize += cnt;

	xab_save();
		/* | time | event | rc | snd_serial | count | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putnint(&snd_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}
int
xab_getstring(cp)
    char *cp;
{
	int rc, cnt;

	rc = getstring(cp);
	xabevent = XAB_getstring;
	cnt = strlen(cp);

	/* tell xab about this call */
	xab_save();
		/* | time | event | rc | rcv_serial | cnt | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putnint(&rcv_serial, 1);
		putnint(&cnt, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

/*********************
* User Library Calls *
*********************/

int
xab_barrier(name, num)
    char *name;     /* name of barrier (any string) */
    int num;        /* number that must reach for us to continue */
{ 
	int tmp;

	/* tell xab about this call */
	xabevent = XAB_barrier;
	xab_save();
		/* | time | event | name | num | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putstring(name);
		putnint(&num, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	tmp = barrier(name, num);

	xabevent = XAB_barrier_done;
	xab_save();
		/* | time | event | rc | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);
}

int
xab_enroll(proc)
    char *proc;
{
	char myhost[MAXHOSTNAMELEN];

	int tmp;

	/* must enroll before telling xab about it */
	tmp = enroll(proc);

	/* find out the host name */
	if (gethostname(myhost, MAXHOSTNAMELEN)) {
		perror("xab_enroll");
		strcpy(myhost, "Unknown");
		}

	/* tell xab about this call */
	xabevent = XAB_enroll;
	initsend();
	xab_puttime();
	putnint(&xabevent, 1);
	putstring(myhost);
	/* | time | XAB_enroll | hostname | */
	snd(XAB, 0, XABMTYPE);

	return(tmp);
}

int
xab_initiate(aout, arch)
    char *aout;     /* name of executable */
    char *arch;     /* architecture or NULL */
{
	int pid;

	/* tell xab that we're calling initiate */
	xabevent = XAB_initiate;
	xab_save();
		/* | time | event | aout | arch | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);

		/* check for null aout string */
		if (aout) { 
			putstring(aout); 
			}
		else { 
			putstring("NULL"); 
			}
	
		/* check for null arch string */
		if (arch) { 
			putstring(arch); 
			}
		else { 
			putstring("NULL"); 
			}

		snd(XAB, 0, XABMTYPE);
	xab_restore();

	/* actually do the initate */
	pid = initiate(aout, arch);

	/* tell xab about this call's completion */
	xabevent = XAB_initiate_done;
	xab_save();
		/* | time | event | pid | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&pid, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(pid);
}

int
xab_initiateM(aout, host)
	char *aout;     /* name of executable */
	char *host;     /* hostname */
{
	int pid;

	/* tell xab that we're calling initiate */
	xabevent = XAB_initiateM;
	xab_save();
		/* | time | event | aout | arch | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);

		/* check for null aout string */
		if (aout) { 
			putstring(aout); 
			}
		else { 
			putstring("NULL"); 
			}
	
		/* check for null host string */
		if (host) { 
			putstring(host); 
			}
		else { 
			putstring("NULL"); 
			}

		snd(XAB, 0, XABMTYPE);
	xab_restore();

	/* actually do the initate */
	pid = initiateM(aout, host);

	/* tell xab about this call's completion */
	xabevent = XAB_initiateM_done;
	xab_save();
		/* | time | event | pid | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&pid, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(pid);
}

void
xab_leave()
{ 
	/* tell xab about this call */
	xabevent = XAB_leave;
	initsend();
	xab_puttime();
	putnint(&xabevent, 1);
	snd(XAB, 0, XABMTYPE);

	leave(); 
}

int
xab_probe(type)
    int type;   /* message type we want or (-1) for wildcard */
{
	/* tell xab about this call */
	
	xabevent = XAB_probe;
	xab_save();
		/* | time | event | probe type | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&type, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(probe(type));
}

int
xab_pstatus(nproc, mixed)
	int *nproc;
	int *mixed;
{
	int rc;
	int t=0;

	xabevent = XAB_pstatus;
	rc = pstatus(nproc, mixed);

	xab_save();
		/* | time | event | rc | nproc | mixed | */
		initsend();
		xab_puttime();
		putnint(&rc, 1);
	        if (nproc!=0) {
		    putnint(nproc, 1);
		}
	        else {
		    putnint(&t,1);
		}
	        if (mixed!=0) {
		    putnint(mixed,1);
		}
	        else {
		    putnint(&t,1);
		}
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

/* xab_rcv(type) 
 * Send a message to xab before doing the user's rcv
 * Do the user's rcv
 * Strip the message serial number off of the user's message
 * Send info about the rcv back to xab
*/

int
xab_rcv(type)
	int type;
{
	int rc;
	int len, instance;
	char component[MAXUPROCNAMELEN];

	/* tell xab about this call */
	xabevent = XAB_rcv;
	xab_save();
		/* | time | event | type | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&type, 1);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	/* rcv the users message */
	rc = rcv(type);
	getnint(&rcv_serial, 1);

	/* send message to xab */
	xabevent = XAB_rcv_done;
	rcvinfo(&len, &type, component, &instance);
	xab_save();
/* | time | event | rc | src_comp | src_pid | msg_type | length | serial | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putstring(component);
		putnint(&instance, 1);
		putnint(&type, 1);
		putnint(&len, 1);
		putnint(&rcv_serial,1);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	return(rc);
}


/* xab_vrcv(type) 
 * Send a message to xab before doing the user's vrcv
 * Do the user's vrcv
 * Strip the message serial number off of the user's message
 * Send info about the vrcv back to xab
*/

int
xab_vrcv(type)
	int type;
{
	int rc;
	int len, instance;
	char component[MAXUPROCNAMELEN];

	/* tell xab about this call */
	xabevent = XAB_vrcv;
	xab_save();
		/* | time | event | type | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&type, 1);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	/* rcv the users message */
	rc = vrcv(type);
	getnint(&rcv_serial, 1);

	/* send message to xab */
	xabevent = XAB_vrcv_done;
	rcvinfo(&len, &type, component, &instance);
	xab_save();
/* | time | event | rc | src_comp | src_pid | msg_type | length | serial | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putstring(component);
		putnint(&instance, 1);
		putnint(&type, 1);
		putnint(&len, 1);
		putnint(&rcv_serial,1);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	return(rc);
}


int
xab_rcvmulti(ntypes, types)
    int ntypes;         /* number of message types */
    int *types;         /* array of message types */
{
	int rc;
	int type, len, instance;
	char component[MAXUPROCNAMELEN];

	/* tell xab about this call */
	xabevent = XAB_rcv_multi;
	xab_save();
		/* | time | event | ntype | types* | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&ntypes, 1);
		putnint(types, ntypes);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	/* rcv the users message */
	rc = rcvmulti(ntypes, types);
	getnint(&rcv_serial, 1);

	/* send message to xab */
	xabevent = XAB_rcv_multi_done;
	rcvinfo(&len, &type, component, &instance);

	xab_save();
/* | time | event | rc | src_comp | src_pid | msg_type | length | serial | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		putstring(component);
		putnint(&instance, 1);
		putnint(&type, 1);
		putnint(&len, 1);
		putnint(&rcv_serial,1);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	return(rc);
}

int
xab_rcvinfo(len, type, proc, inum)
    int *len, *type, *inum;
    char *proc;
{
	int rc;

	rc = rcvinfo(len, type, proc, inum);
	xabevent = XAB_rcvinfo;

	xab_save();
		initsend();
		xab_puttime();
		putnint(&xabevent,1);
		putnint(&rc, 1);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	return(rc);

}

int
xab_ready(event)
	char *event;
{
	int rc;
	long time[2];
	struct timeval tv;
	struct timezone tz;

	if(gettimeofday(&tv, &tz)) {
	    fprintf(stderr, "gettimeofday() choked, bailing\n");
	    exit(-1);
        }
	rc = ready(event);
	xabevent = XAB_ready;

	xab_save();
		initsend();
    	time[0] = tv.tv_sec;
    	time[1] = tv.tv_usec;
    	putnlong(time,2);
		putnint(&xabevent,1);
		putnint(&rc, 1);
		snd(XAB, 0, XABMTYPE);	
	xab_restore();

	return(rc);
}

/*
*	xab_snd
*	send off the message
*	send the headers to the xab process
*/

int
xab_snd(proc, inum, type)
    char *proc;
    int inum;
    int type;
{
	int i, tmp;
	mdesc *t;
	long time[2];
	struct timeval tv;
	struct timezone tz;

	if(gettimeofday(&tv, &tz)) {
	    fprintf(stderr, "gettimeofday() choked, bailing\n");
	    exit(-1);
        }
	/* we have to send the message off before telling xab */
	tmp = snd(proc, inum, type);

	/* tell xab about this call */
	xabevent = XAB_snd;
	xab_save();
	/* | time | XAB_snd | rc | dst_comp | dst_pid | msg_type | serial | length */
	/* length is in bytes before xdr */
	initsend();
    	time[0] = tv.tv_sec;
    	time[1] = tv.tv_usec;
    	putnlong(time,2);
	        putnint(&xabevent, 1);
	        putnint(&tmp, 1);
	        putstring(proc);
		putnint(&inum, 1);
		putnint(&type, 1);
		putnint(&snd_serial,1);
		putnint(&xab_msize,1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);

}

/*
*	xab_vsnd
*	send off the message
*	send the headers to the xab process
*/

int
xab_vsnd(proc, inum, type)
    char *proc;
    int inum;
    int type;
{
	int i, tmp;
	mdesc *t;
	long time[2];
	struct timeval tv;
	struct timezone tz;

	if(gettimeofday(&tv, &tz)) {
	    fprintf(stderr, "gettimeofday() choked, bailing\n");
	    exit(-1);
        }
	/* we have to send the message off before telling xab */
	tmp = vsnd(proc, inum, type);

	/* tell xab about this call */
	xabevent = XAB_vsnd;
	xab_save();
/* | time | XAB_vsnd | rc | dst_comp | dst_pid | msg_type | serial | length */
		/* length is in bytes before xdr */
	initsend();
    	time[0] = tv.tv_sec;
    	time[1] = tv.tv_usec;
    	putnlong(time,2);
		putnint(&xabevent, 1);
		putnint(&tmp, 1);
		putstring(proc);
		putnint(&inum, 1);
		putnint(&type, 1);
		putnint(&snd_serial,1);
		putnint(&xab_msize,1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(tmp);

}

int
xab_status(proc, inum)
    char *proc;
    int inum;
{
	int rc;

	rc = status(proc, inum);
	xabevent = XAB_status;

	xab_save();
		initsend();
		xab_puttime();
		putnint(&xabevent,1);
		putnint(&rc, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_terminate(proc, inum)
    char *proc;
    int inum;
{
	int rc;
	long time[2];
	struct timeval tv;
	struct timezone tz;

	if(gettimeofday(&tv, &tz)) {
	    fprintf(stderr, "gettimeofday() choked, bailing\n");
	    exit(-1);
        }

	xabevent = XAB_terminate;
	rc = terminate(proc, inum);

	xab_save();
		initsend();
    	time[0] = tv.tv_sec;
    	time[1] = tv.tv_usec;
    	putnlong(time,2);
		putnint(&xabevent,1);
		putstring(proc);
		putnint(&inum, 1);
		putnint(&rc, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);

}

int
xab_waituntil(event)
    char *event;
{
	int rc;

	xabevent = XAB_waituntil;
	xab_save();
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putstring(event);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	rc = waituntil(event);

	xabevent = XAB_waituntil_done;
	xab_save();
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putnint(&rc, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}

int
xab_whoami(proc, inum)
    char *proc;
    int *inum;
{
	char myhost[MAXHOSTNAMELEN];
	int rc;

	/* must call whoami before telling xab about it */
	rc = whoami(proc, inum);


	/* find out the host name */
	if (gethostname(myhost, MAXHOSTNAMELEN)) {
		perror("xab_enroll");
		strcpy(myhost, "Unknown");
		}

	/* tell xab about this call */
	xabevent = XAB_whoami;
	xab_save();
		/* | time | XAB_whoami | myhost | rc | */
		initsend();
		xab_puttime();
		putnint(&xabevent, 1);
		putstring(myhost);
		putnint(&rc, 1);
		snd(XAB, 0, XABMTYPE);
	xab_restore();

	return(rc);
}
